;;;
;;; dskimage.lisp
;;;
;;; Read disk images used by Apple II emulators.
;;;
;;; $Id: dskimage.lisp,v 1.1 2000/01/21 23:04:51 jaoswald Exp $
;;;
;;; $Log: dskimage.lisp,v $
;;; Revision 1.1  2000/01/21 23:04:51  jaoswald
;;; Initial revision
;;;
;;;

;;; need code to handle .NIB files?
;;;    Catakig .DSK files are gzip'ed .NIB images
;;;    My gzip code seems to handle the unzipping fine, but I can't
;;;    do anything with the .NIB files.
;;;
;;; If I were to do that, I would pretty easily be able to create the 
;;; tools to "normalize" some .NIB files, like the Infocom games,
;;; to DOS 3.3 format.
;;;
 
(defconstant +diskii-tracks+ 35)
(defconstant +dos33-sectors+ 16 "DOS 3.3 sectors per Disk II track.")
(defconstant +dos33-sector-length+ 256 "Length in bytes of a DOS 3.3 sector.")
(defconstant +dos33-image-length+ (* +diskii-tracks+ +dos33-sectors+
				     +dos33-sector-length+))

(defconstant +dos33-directory-length+ 15 
  "Standard length of a DOS 3.3 directory in sectors")

(defconstant +dos33-vtoc-track+ #x11 
  "The Disk II track on which the Volume Table of Contents is located.")

(defconstant +dos33-vtoc-sector+ #x0 
  "The Disk II sector at which the Volume Table of Contents is located.")

(defconstant +vtoc-dirtrack-offset+ 1 "The VTOC contains, at this offset,
the track of the first directory sector. Furthermore, directory sectors 
contain, at this offset, the track of the next directory sector, or 
zero if there is no next directory sector.")

(defconstant +vtoc-dirsector-offset+ 2 "The VTOC contains, at this offset, 
the sector of the first directory. Furthermore, directory sectors
contain, at this offset, the sector of the next directory sector." )

(defconstant +vtoc-tslistlength-offset+ #x27
  "The VTOC contains, at this offset, the number of track/sector entries 
contained in a single sector of a T/S list.")

(defconstant +vtoc-dosversion-offset+ 3
  "The VTOC contains, at this offset, a number indicating the version of
DOS. For DOS 3.3, this contains 3. For 3.2, it contains 2.")

(defconstant +vtoc-volume-offset+ 6
  "Nominally, the VTOC contains, at this offset, the volume number of the
disk.")

(defconstant +vtoc-nexttrackuse-offset+ 48 "Next track from which to 
allocate space.")

(defconstant +vtoc-trackusedirection-offset+ 49 "Incremental direction 
for allocation of tracks.")

;;; description of byte locations in the VTOC sector under DOS 3.3
(defconstant +vtoc-trackcount-offset+ 52 "Number of tracks on the disk.")
(defconstant +vtoc-sectorcount-offset+ 53 "Number of sectors per track.")
(defconstant +vtoc-sectorsize-low-offset+ 54)
(defconstant +vtoc-sectorsize-high-offset+ 55)

(defconstant +vtoc-allocationmap-offset+ 56)
(defconstant +vtoc-bytes-per-track+ 4 "Number of bytes used for allocation
 map. Note that the third and fourth bytes are assigned, but not used.")


(defclass disk-image ()
  ())

(defclass dos-image (disk-image)
  ((data :accessor data :initarg :data)))

(defun read-dos33-image-stream (stream)
  (let ((data (make-array +dos33-image-length+ :element-type 'unsigned-byte)))
    (dotimes (i +dos33-image-length+)
      (setf (aref data i) (read-byte stream)))
    (make-instance 'dos-image :data data)))

(defun read-dos33-image (pathname)
  (with-open-file (str pathname :direction :input
                       :element-type '(unsigned-byte 8))
    (read-dos33-image-stream str)))

(defmethod track-sector ((image dos-image) track sector &key &allow-other-keys)
  (when (or (< track 0) (>= track +diskii-tracks+)
	    (< sector 0) (>= sector +dos33-sectors+))
    (error "Illegal track ~D sector ~D" track sector))
  (let ((offset (* (+ (* track +dos33-sectors+) sector) 
		   +dos33-sector-length+)))
    (make-array +dos33-sector-length+ :displaced-to (data image)
		:displaced-index-offset offset)))

(defconstant +prodos-block-sequence+
  '(0 2 4 6 8 #xA #xC #xE 1 3 5 7 9 #xB #xD #xF)
  "Order of physical sector usage on a Disk II floppy disk under ProDos.")

(defconstant +prodos-order+
  '(0 #xE #xd #xc #xb #xa #x9 #x8 #x7 #x6 #x5 #x4 #x3 #x2 #x1 #xf)
  "Suspected order in .dsk image. Agrees with comment in Applewin's Aw_image.cpp.")

(defconstant +prodos-block-length+ 512
  "Bytes in a ProDOS block.")

(defconstant +prodos-disk-II-blocks+ #x118
  "Number of ProDOS blocks in a single Disk II floppy")
(defconstant +prodos-disk-II-blocks-track+ (/ +dos33-sectors+ 2)
  "Number of ProDOS blocks in a Disk II track.")

(defun prodos-disk-II-sectors (block)
  "Returns four values: the track and physical sector of the first sector
and the track and physical sector of the second sector of a ProDOS block."
  (multiple-value-bind (track trk-block)
      (floor block +prodos-disk-II-blocks-track+)
    (let ((first-sector (nth (* 2 trk-block) +prodos-order+))
	  (second-sector (nth (1+ (* 2 trk-block)) +prodos-order+)))
      (values track first-sector track second-sector))))

(defmethod prodos-block ((image dos-image) block-number)
  (unless (and (<= 0 block-number)
	       (< block-number +prodos-disk-II-blocks+))
    (error "ProDOS block number out of range."))
  (multiple-value-bind (t1 s1 t2 s2)
      (prodos-disk-II-sectors block-number)
    (concatenate 'vector (track-sector image t1 s1)
		 (track-sector image t2 s2))))

(defconstant +prodos-volume-directory-key-block+ 2 
  "Disk block containing the first block of the root directory.")

(defmethod prodos-directory-image ((image disk-image) &optional 
				   (starting-block 
				    +prodos-volume-directory-key-block+))
  (let ((d (make-array 0 :fill-pointer 0 :adjustable t)))
    (do ((b starting-block)
	 (last-block 0))
	((= b 0) d)
      (let ((block (prodos-block image b)))
	(vector-push-extend block d)
	(let ((prev-block (+ (aref block 0) (* 16 (aref block 1))))
	      (next-block (+ (aref block 2) (* 16 (aref block 3)))))
	  (unless (= prev-block last-block)
	    (error "Doubly-linked directory chain broken after block ~D. Current-block ~D Prev-block ~D, next-block ~D."
		   last-block b prev-block next-block))
	  (setf last-block b)
	  (setf b next-block))))))

(defclass prodos-volume-directory-header ()
  ((volume-name :accessor volume-name :initarg :volume-name)
   (creation-time :accessor creation-time :initarg :creation-time)
   (creation-version :accessor creation-version :initarg :creation-version)
   (min-version :accessor min-version :initarg :min-version)
   (access-flags :accessor access-flags :initarg :access-flags)
   (entry-length :accessor entry-length :initarg :entry-length)
   (entries-per-block :accessor entries-per-block :initarg :entries-per-block)
   (file-count :accessor file-count :initarg :file-count)
   (volume-bitmap-block :accessor volume-bitmap-block :initarg 
			:volume-bitmap-block)
   (volume-block-count :accessor volume-block-count :initarg
		       :volume-block-count)))

(defun prodos-universal-time (ymd-byte0 ymd-byte1 hms-byte0 hms-byte1
			      &optional (year-offset 1900))
  "Returns the universal time corresponding to the four bytes of a ProDOS
1.0 date & time entry."
  (let ((year-bits (ldb (byte 7 1) ymd-byte1))
	(month-bits (+ (* 8 (ldb (byte 1 0) ymd-byte1))
		       (ldb (byte 3 5) ymd-byte0)))
	(day-bits (ldb (byte 5 0) ymd-byte0))
	(hours hms-byte1)
	(minutes hms-byte0))
    (encode-universal-time
     0 minutes hours day-bits month-bits (+ year-offset year-bits))))

(defconstant +vdh-storage-type+ #xF
  "Storage type code for a volume directory header.")
(defconstant +vdh-st-name-offset+ #x4
  "Offset within volume directory header of the volume storage type, name length")

(defconstant +vdh-name-offset+ #x5
  "Offset within volume directory header of the volume name.")
(defconstant +vdh-creation-offset+ #x1C
  "Offset within volume directory header of the creation date/time.")
(defconstant +vdh-creation-version-offset+ #x20
  "Offset within volume directory header of the creation ProDOS version.")
(defconstant +vdh-min-version-offset+ #x21
  "Offset within volume directory header of the minimum compatible ProDOS version.")
(defconstant +vdh-access-offset+ #x22
  "Offset within volume directory header of the volume access flags.")
(defconstant +vdh-entry-length-offset+ #x23
  "Offset within volume directory header of the vol directory entry length.")
(defconstant +vdh-entry-count-offset+ #x24
  "Offset within volume directory header of the entry count per vol dir block.")
(defconstant +vdh-file-count-offset+ #x25
  "Offset within volume directory header of the volume active file count.")
(defconstant +vdh-bitmap-offset+ #x27
  "Offset within volume directory header of the block containing the volume bit map.")
(defconstant +vdh-block-count-offset+ #x29
  "Offset within volume directory header of the volume block count.")


(defun decode-prodos-volume-header (dir-image)
  (let ((header-block (aref dir-image 0)))
    (let ((storage-type (ldb (byte 4 4) (aref header-block +vdh-st-name-offset+)))
	  (name-length (ldb (byte 4 0) (aref header-block +vdh-st-name-offset+)))
	  (creation-time (prodos-universal-time
			  (aref header-block +vdh-creation-offset+)
			  (aref header-block (+ 1 +vdh-creation-offset+))
			  (aref header-block (+ 2 +vdh-creation-offset+))
			  (aref header-block (+ 3 +vdh-creation-offset+))))
	  (creation-version (aref header-block +vdh-creation-version-offset+))
	  (min-version (aref header-block +vdh-min-version-offset+))
	  (access (aref header-block +vdh-access-offset+))
	  (entry-length (aref header-block +vdh-entry-length-offset+))
	  (entry-per-block (aref header-block +vdh-entry-count-offset+))
	  (file-count (+ (aref header-block +vdh-file-count-offset+)
			 (* 256 (aref header-block 
				      (+ 1 +vdh-file-count-offset+)))))
	  (bitmap-block (+ (aref header-block +vdh-bitmap-offset+)
			 (* 256 (aref header-block 
				      (+ 1 +vdh-bitmap-offset+)))))
	  (block-count (+ (aref header-block +vdh-block-count-offset+)
			  (* 256 (aref header-block 
				       (+ 1 +vdh-block-count-offset+))))))
      (let ((name (make-string name-length)))
	(dotimes (i name-length)
	  (setf (aref name i)
		(apple-code-char 
		 (aref header-block (+ i +vdh-name-offset+)))))

	(unless (= storage-type +vdh-storage-type+)
	  (warn "Storage type does not match Volume Directory Header"))

	(make-instance 
	 'prodos-volume-directory-header
	 :volume-name name
	 :creation-time creation-time
	 :creation-version creation-version
	 :min-version min-version
	 :access-flags access
	 :entry-length entry-length
	 :entries-per-block entry-per-block
	 :file-count file-count
	 :volume-bitmap-block bitmap-block
	 :volume-block-count block-count)))))

(defmethod vtoc ((image dos-image))
  (track-sector image +dos33-vtoc-track+ +dos33-vtoc-sector+))

(defclass prodos-directory-entry ()
  ((file-name :accessor file-name :initarg :file-name)
   (storage-type :accessor storage-type :initarg :storage-type)
   (file-type :accessor file-type :initarg :file-type)
   (aux-type :accessor aux-type :initarg :aux-type)
   (file-blocks :accessor file-blocks :initarg :file-blocks)
   (file-length :accessor file-byte-length :initarg :file-length)
   (key-pointer :accessor key-pointer :initarg :key-pointer)
   (dir-pointer :accessor dir-pointer :initarg :dir-pointer)
   (creation-time :accessor creation-time :initarg :creation-time)
   (modify-time :accessor modify-time :initarg :modify-time)
   (creation-version :accessor creation-version :initarg :creation-version)
   (min-version :accessor min-version :initarg :min-version)
   (access-flags :accessor access-flags :initarg :access-flags)))

(defconstant +prodos-ptrs-length+ 4 
  "Number of bytes in a directory block consumed by block pointers.")

(defconstant +prodos-name-offset+ 1 
  "Offset within a ProDOS directory entry of the name.")
(defconstant +prodos-type-offset+ #x10
  "Offset within a ProDOS directory entry of the file type.")
(defconstant +prodos-keyptr-offset+ #x11
  "Offset within a ProDOS directory entry of the file key pointer.")
(defconstant +prodos-blkcnt-offset+ #x13
  "Offset within a ProDOS directory entry of the file blocks-used count.")
(defconstant +prodos-bytelen-offset+ #x15
  "Offset within a ProDOS directory entry of the file length in bytes.")
(defconstant +prodos-creation-offset+ #x18
  "Offset within a ProDOS directory entry of the file creation time.")
(defconstant +prodos-modify-offset+ #x21
  "Offset within a ProDOS directory entry of the file modification time.")
(defconstant +prodos-version-offset+ #x1c
  "Offset within a ProDOS directory entry of the ProDOS creation version.")
(defconstant +prodos-minversion-offset+ #x1d
  "Offset within a ProDOS directory entry of the minimum compatible 
ProDOS version.")
(defconstant +prodos-access-offset+ #x1e
  "Offset within a ProDOS directory entry of the file access flags.")
(defconstant +prodos-auxtype-offset+ #x1f
  "Offset within a ProDOS directory entry of the ProDOS creation version.")
(defconstant +prodos-dirptr-offset+ #x25
  "Offset within a ProDOS directory entry of the parent directory pointer.")



(defun read-prodos-directory-entry (dir-image entry-num)
  "Dir-image is an array of blocks making up the directory.
Entry-num starts at zero, does not include the directory header."
  (let* ((dir-header (decode-prodos-volume-header dir-image))
	 (entry-length (entry-length dir-header))
	 (entries-per-block (entry-length dir-header)))
    (multiple-value-bind (blk num)
	(floor (+ 1 entry-num) ; must count directory header
	       entries-per-block)
      (let ((block (aref dir-image blk))
	    (offset (+ +prodos-ptrs-length+ (* num entry-length))))
	(let ((storage-type (ldb (byte 4 4) (aref block offset)))
	      (name-length (ldb (byte 4 0) (aref block offset)))
	      (file-type (aref block (+ offset +prodos-type-offset+)))
	      (key-pointer (+ (aref block (+ offset +prodos-keyptr-offset+))
			      (* 256 (aref block (+ 1 offset 
						    +prodos-keyptr-offset+)))))
	      (blocks-used (+ (aref block (+ offset +prodos-blkcnt-offset+))
			      (* 256 (aref block 
					   (+ 1 offset 
					      +prodos-blkcnt-offset+)))))
	      (file-length (+ (aref block (+ offset +prodos-bytelen-offset+))
			      (* 256 (aref block 
					   (+ 1 offset 
					      +prodos-bytelen-offset+)))
			      (* 65536 (aref block 
					     (+ 2 offset 
						+prodos-bytelen-offset+)))))
	      (creation-time (prodos-universal-time
			      (aref block (+ offset +prodos-creation-offset+))
			      (aref block (+ 1 offset +prodos-creation-offset+))
			      (aref block (+ 2 offset +prodos-creation-offset+))
			      (aref block (+ 3 offset +prodos-creation-offset+))))
	      (modify-time (prodos-universal-time
			      (aref block (+ offset +prodos-modify-offset+))
			      (aref block (+ 1 offset +prodos-modify-offset+))
			      (aref block (+ 2 offset +prodos-modify-offset+))
			      (aref block (+ 3 offset +prodos-modify-offset+))))
	      (prodos-version (aref block (+ offset +prodos-version-offset+)))
	      (min-version (aref block (+ offset +prodos-minversion-offset+)))
	      (access (aref block (+ offset +prodos-access-offset+)))
	      (aux-type (+ (aref block (+ offset +prodos-auxtype-offset+))
			   (* 256 (aref block
					(+ 1 offset +prodos-auxtype-offset+)))))
	      (dir-pointer (+ (aref block (+ offset +prodos-dirptr-offset+))
			      (* 256 (aref block
					   (+ 1 offset +prodos-dirptr-offset+))))))
	  (let ((name (make-string name-length)))
	    (dotimes (i name-length)
	      (setf (aref name i) (apple-code-char 
				   (aref block (+ offset +prodos-name-offset+
						  i)))))
	    
	    (make-instance
	     'prodos-directory-entry
	     :file-name name
	     :file-type file-type
	     :storage-type storage-type
	     :aux-type aux-type
	     :file-blocks blocks-used
	     :file-length file-length
	     :key-pointer key-pointer
	     :dir-pointer dir-pointer
	     :creation-time creation-time
	     :modify-time modify-time
	     :creation-version prodos-version
	     :min-version min-version
	     :access-flags access)))))))
	   			
(defconstant +prodos-st-deleted+ 0 "Storage type of a deleted file.")
(defconstant +prodos-st-seedling+ 1 
  "Storage type of a single block 'seedling' file. The key block is the file contents.")
(defconstant +prodos-st-sapling+ 2 
  "Storage type of a 'sapling' file. The key block is a single block listing the file's constituent blocks.")
(defconstant +prodos-st-tree+ 3
  "Storage type of a 'tree' file. The key block is a list of blocks each containing a list of constituent blocks.")
      
(defmethod file-block-list ((image disk-image) 
			    (direntry prodos-directory-entry))
  "Returns a list containing the block numbers making up the file described by
the directory entry. Nil is returned for a deleted file. Nils in the list represent unallocated blocks."

  (let ((st (storage-type direntry)))
    (cond
      ((= st +prodos-st-deleted+) nil) ; deleted file
      ((= st +prodos-st-seedling+) (list (key-pointer direntry)))
      ((= st +prodos-st-sapling+)
       (let ((index-block (prodos-block image (key-pointer direntry)))
	     (blkcount (ceiling (file-byte-length direntry) 
				+prodos-block-length+))
	     (block-list nil))
	     ;; block-count counts only allocated blocks, including
	     ;; the index block, but not unallocated blocks.
	 (dotimes (i blkcount (nreverse block-list))
	   (let ((block (+ (aref index-block i)
			   (* 256 (aref index-block (+ i 256))))))
	     (push (if (zerop block)
		       nil
		       block)
		   block-list)))))
      ((= st +prodos-st-tree+)
       (error "Tree files not yet implemented."))
      (t (error "Not a file.")))))

(defmethod raw-file-data ((image disk-image) 
			  (direntry prodos-directory-entry))
  (let* ((block-list (file-block-list image direntry))
	 (flen (file-byte-length direntry))
	 (binary-array (make-array flen))
	 (offset 0))
    (unless (null block-list)
      (dolist (b block-list binary-array)
	(let ((blk (prodos-block image b)))
	  (dotimes (i (min +prodos-block-length+ (- flen offset)))
	    (setf (aref binary-array (+ offset i))
		  (aref blk i)))
	  (incf offset +prodos-block-length+))))))

(defmethod directory-image ((image dos-image))
  (let ((v (vtoc image))
	(d (make-array +dos33-directory-length+ :fill-pointer 0 
		       :adjustable t)))

    ;;; directory sectors are actually a linked-list. 
    ;;; On a standard disk, it is always stored in the same way, but 
    ;;; we'll follow the rules.

    (do ((track (aref v +vtoc-dirtrack-offset+))
	 (sector (aref v +vtoc-dirsector-offset+)))
	((= track 0) d)
	(let ((s (track-sector image track sector)))
	  (vector-push-extend s d)
	  (setf track (aref s +vtoc-dirtrack-offset+)
		sector (aref s +vtoc-dirsector-offset+))))))
	
(defclass directory-entry ()
  ((file-name :accessor file-name :initarg :file-name)
   (file-type :accessor file-type :initarg :file-type)
   (file-locked :accessor file-locked :initarg :file-locked)
   (ts-track :accessor ts-track :initarg :ts-track)
   (ts-sector :accessor ts-sector :initarg :ts-sector)
   (file-size :accessor file-size :initarg :file-size)))

(defclass deleted-file-entry (directory-entry) ())
 
(defconstant +directory-entry-offset+ 11)
(defconstant +directory-entry-length+ 35)
(defconstant +directory-entries-per-sector+ 7)
(defconstant +file-name-length+ 30)
(defconstant +file-name-offset+ 3)
(defconstant +file-tstrack-offset+ 0)
(defconstant +file-tssector-offset+ 1)
(defconstant +file-type-offset+ 2)
(defconstant +file-size-low-offset+ 33)
(defconstant +file-size-high-offset+ 34)

(defun apple-code-char (code)
  (code-char (logand 127 code)))

(defun apple-file-type (code)
  (let ((type-code (logand code 127))
	(locked (logbitp 7 code)))
    (values 
     (case type-code
	   (0 :text)
	   (1 :integer)
	   (2 :applesoft)
	   (4 :binary)
	   (8 :s-type)
	   (16 :r-type)
	   (32 :a-type)
	   (64 :b-type)
	   (t type-code))
     locked)))

;;; if ts-track is 0, not a valid entry (past end of directory)
;;; if ts-track is #xFF, the file has been deleted

(defun read-directory-entry (dir-image entry-num)
  (multiple-value-bind (s cnt)
		       (floor entry-num +directory-entries-per-sector+)
     (let ((sec (aref dir-image s))
	   (offset (+ +directory-entry-offset+ (* +directory-entry-length+
						  cnt)))
	   (name (make-string +file-name-length+)))
       (dotimes (i +file-name-length+)
	  (setf (aref name i) (apple-code-char (aref sec (+ offset 
							    +file-name-offset+
							    
							    i)))))
       (multiple-value-bind (file-type locked)
	    (apple-file-type (aref sec (+ offset +file-type-offset+))) 
	  (let ((file-size (+ (* 256 (aref sec (+ offset 
						  +file-size-high-offset+)))
			      (aref sec (+ offset +file-size-low-offset+)))))
				 
            ;; should check for deleted file, or null file entry.

	    (make-instance 'directory-entry 
			   :file-name name
			   :file-type file-type
			   :file-locked locked
			   :file-size file-size
			   :ts-track (aref sec (+ offset 
						  +file-tstrack-offset+))
			   :ts-sector (aref sec (+ offset 
						 +file-tssector-offset+))))))))

(defconstant +tsl-tsltrack-offset+ 1 "Track of next portion of T/S list.")
(defconstant +tsl-tslsector-offset+ 2 "Sector of next portion of T/S list.")

(defconstant +tsl-filesector-low-offset+ 5 "Low byte of the sector count 
in the file corresponding to the first entry of this portion of the T/S list.")

(defconstant +tsl-filesector-high-offset+ 6 "High byte of the sector count 
in the file corresponding to the first entry of this portion of the T/S list.")

(defconstant +tsl-tsentry-offset+ 12 "First track entry in this portion 
of the T/S list.")

(defconstant +tsl-entries-per-sector+ 122 "For DOS 3.3, the number of 
T/S entries in a one-sector portion of a T/S list.")

(defclass tslist ()

  ;; a pair of arrays, containing a sequence of tracks and corresponding
  ;; sector offsets, which together form a sequence of disk sectors that
  ;; make up a data stream

  ((tracks :accessor tracks :initarg :tracks)
   (sectors :accessor sectors :initarg :sectors)))

(defclass raw-tslist (tslist)

  ;; a raw DOS 3.3 track/sector list, containing an array of the raw
  ;; disk sectors that made up the T/S list for a file.

  ((raw-data :accessor raw-data :initarg :raw-data)))

(defmethod sector-count ((ts tslist))
  (length (sectors ts)))


(defun get-tslist (image directory-entry)
  (let ((tsl (make-array 1 :adjustable t :fill-pointer 0))
        (seccnt (file-size directory-entry))
        (tsllength 0))
    
    (do ((trk (ts-track directory-entry))
         (sec (ts-sector directory-entry)))
        ((= trk 0))
      
      (let ((data (track-sector image trk sec)))
        (vector-push-extend data tsl)
        (incf tsllength)

        ;; follow link to next portion
        
        (setf trk (aref data +tsl-tsltrack-offset+)
              sec (aref data +tsl-tslsector-offset+))))
    
    (let* ((entries (- seccnt tsllength))
           (tracks (make-array entries))
           (sectors (make-array entries)))

      (dotimes (s entries)
        (multiple-value-bind (tssec index)
                             (floor s +tsl-entries-per-sector+)
          (let ((secdata (aref tsl tssec))
                (offset (* 2 index)))
            (setf (aref tracks s) 
                  (aref secdata (+ +tsl-tsentry-offset+ offset))
                  (aref sectors s) 
                  (aref secdata 
                        (+ 1 +tsl-tsentry-offset+ offset))))))
      
      (make-instance 'raw-tslist :raw-data tsl 
                     :tracks tracks :sectors sectors))))

(defmethod raw-file-data ((image dos-image) (ts tslist))
  "Return an array containing the binary disk data from the 
sectors enumerated in TS."

  (let* ((seccnt (sector-count ts))
         (tracks (tracks ts))
         (sectors (sectors ts))
         (binary-array (make-array (* +dos33-sector-length+ 
                                      seccnt))))
    (dotimes (s seccnt binary-array)
      (let ((sector-data (track-sector image 
                                       (aref tracks s)
                                       (aref sectors s)))
            (offset (* s +dos33-sector-length+)))

        (dotimes (i +dos33-sector-length+)
          (setf (aref binary-array (+ offset i))
                (aref sector-data i)))))))

(defun binary-transfer-file (disk-image directory-entry
                                        output-stream)

  (labels ((transfer-sector (sec)
             (dotimes (i +dos33-sector-length+)
               (write-byte (aref sec i) output-stream))))

  (let* ((tsl (get-tslist disk-image directory-entry))
         (trks (tracks tsl))
         (secs (sectors tsl))
         (seccnt (length trks)))

    (dotimes (s seccnt)
      (transfer-sector (track-sector disk-image 
                                     (aref trks s) (aref secs s)))))))

(defmethod binary-file-array ((image dos-image) (de directory-entry))

  "Returns two values:
an array containing the binary data of a DOS 3.3 binary file described by DE,
and the address at which the file is intended to be loaded."

  (let* ((tsl (get-tslist image de))
         (bdata (raw-file-data image tsl)))

    (let ((bload-address (+ (aref bdata 0)
                            (* 256 (aref bdata 1))))
          (bload-length (+ (aref bdata 2)
                           (* 256 (aref bdata 3)))))
      (values (make-array bload-length
                          :displaced-to bdata 
                          :displaced-index-offset 4)
              bload-address)))) 
                          

    
  
       
    

	     
